/*
 * Himax_platform_i2c.c
 *
 *      Author: 903621
 */

#include "himax_platform.h"
#include "himax_touch_driver.h"
#ifndef UNITTEST
#include "i2c_client.h"
#else
#include "i2c_header.h"
#endif


/* Note: There is no address increment */
int himax_i2c_write(int fd, uint8_t addr, uint8_t *w_data, int w_len)
{
    uint8_t *buf;
    int count;
    int ret;
    bool retry = false;

    if (fd == -1)
        return -EINVAL;

    count = w_len + 1;
    buf = malloc(count);
    if (!buf)
        return -ENOMEM;

    buf[0] = addr;
    memcpy(&buf[1], w_data, w_len);

retry_write:
    ret = i2c_write(fd, buf, count);
    if (ret != count) {
        if (!retry) {
            if ((usleep(20 * 1000)) == 0) {
                retry = true;
                goto retry_write;
            } else {
                ret = errno;
            }
        } else {
            mtouch_error("Himax", "i2c write failed, ret = %d \n", ret);
            ret = -EIO;
        }
    } else {
        ret = 0;
    }

    free(buf);
    return ret;
}

int himax_i2c_combined_writeread(int fd, uint8_t addr, uint8_t *r_data, int r_len)
{
    uint8_t write_buf[1];
    int ret;
    bool retry = false;

    if (fd == -1)
        return -EINVAL;

    write_buf[0] = addr;

retry_read:
    ret = i2c_combined_writeread(fd, write_buf, 1, r_data, r_len);
    if (ret != r_len) {
        if (!retry) {
            if ((usleep(20 * 1000)) == 0) {
                retry = true;
                goto retry_read;
            } else {
                ret = errno;
            }
        } else {
            mtouch_error("Himax", "himax i2c read fail! %d\n", ret);
            ret = -EIO;
        }
    } else {
        ret = 0;
    }

    return ret;

}

int himax_register_read(int fd, uint8_t *read_addr, int read_len, uint8_t *read_data)
{
    uint8_t tmp_addr[4];
    tmp_addr[0] = read_addr[0] & 0xFF;
    tmp_addr[1] = read_addr[1] & 0xFF;
    tmp_addr[2] = read_addr[2] & 0xFF;
    tmp_addr[3] = read_addr[3] & 0xFF;
    uint8_t tmp_buf[1] = {0};

    if (fd == -1)
        return -EINVAL;

    if (himax_i2c_write(fd, 0x00, tmp_addr, 4) != EOK) {
        mtouch_error("Himax", "himax_write fail!\n");
        return EIO;
    }

    if (himax_i2c_write(fd, 0x0C, tmp_buf, 1) != EOK) {
        mtouch_error("Himax", "himax_write fail!\n");
        return EIO;
    }

    if (himax_i2c_combined_writeread(fd, 0x08, read_data, read_len) != EOK) {
        mtouch_error("Himax", "himax_i2c_combined_writeread i2c access fail!\n");
        return EIO;
    }

    //mtouch_info("Himax", "read_data[0]: 0x%x, read_data[1]: 0x%x, read_data[2]: 0x%x, read_data[3]: 0x%x\n", read_data[0], read_data[1], read_data[2], read_data[3]);

    return EOK;
}

int himax_register_write(int fd, uint8_t *write_addr, uint8_t *write_data)
{
    uint8_t tmp_data[8];

    if (fd == -1)
        return -EINVAL;

    tmp_data[0] = write_addr[0] & 0xFF;
    tmp_data[1] = write_addr[1] & 0xFF;
    tmp_data[2] = write_addr[2] & 0xFF;
    tmp_data[3] = write_addr[3] & 0xFF;

    tmp_data[4] = write_data[0] & 0xFF;
    tmp_data[5] = write_data[1] & 0xFF;
    tmp_data[6] = write_data[2] & 0xFF;
    tmp_data[7] = write_data[3] & 0xFF;

    if (himax_i2c_write(fd, 0x00, tmp_data, 8) != EOK) {
        mtouch_error("Himax", "himax_adap_write fail!\n");
        return EIO;
    }
    return 0;
}
